'use strict';

import { getLang } from "../../lang.js";


/** Блок взаимодействия с jssip */
export function CallWindow(options){
  return {
    data() {
      return {
        lang: getLang(CB.globals.config["lang"] == "russian" ? CB.globals.config["lang"] : "english"),
        connected: false,
        prefix: options.prefix,
        name: options.name,
        timer_id: null,
        options: {
          pcConfig: {
            hackStripTcp: true, // Важно для хрома, чтоб он не тупил при звонке
            //rtcpMuxPolicy: 'negotiate', // Важно для хрома, чтоб работал multiplexing. Эту штуку обязательно нужно включить на астере.
            iceServers: [{
              urls:['stun:stun.l.google.com:19302']
            }],
            iceTransportPolicy: "all"
          },

          mediaConstraints: {
            audio: true, // Поддерживаем только аудио
            video: false
          },

          rtcOfferConstraints: {
            offerToReceiveAudio: 1, // Принимаем только аудио
            offerToReceiveVideo: 0
          }
        },
      }
    },

    async mounted() {
      // Получаем данные настроек модуля
      let settings_data = await this.getData(`../../api/dev/settings/cbext.clientbase.${options.api_name}`);
      // получаем данные аккаунтов из настроек
      let account_data = settings_data.data.attributes.value.personal_account_details;

      // устанавливаем данные пользователя телефонии для текущего пользователя
      let user_settings = settings_data.data.attributes.value.users.filter(item => item.cb_user == CB.globals.user.id)[0];
      // проверка на существование пользовательских настроек телефонии для текущего пользователя, иначе выдаём предупреждение и не авторизуем
      if (user_settings == void 0) {
        this.setStoreValue('user_status', {
          status: false,
          message: this.lang["user_not_added_to_telephony"],
        });
        return;
      }

      this.setStoreValue('user_status', {
        status: true,
        message: '',
      });

      this.setStoreValue('provider_settings', {
        server: `${account_data.server}:${account_data.port}`,
        domain: account_data.domain,
        url: options.telephony_api_url, // адрес API
        rlogin: account_data.loginPBX,       // логин для доступа к API
        rkey: account_data.rkey,         // секретный ключ для формирования подписи к запросам
        login: account_data.loginPBX,        // логин личного кабинета
        password: account_data.passwordPBX,     // пароль личного кабинета
        sip_login: user_settings.sip_number,    // логин для авторизации SIP-телефона
        sip_password: user_settings.sip_password, // пароль для авторизации SIP-телефона
      });

      // получаем всех пользователей конфигурации
      let cb_users_data = await this.getData(`../../api/dev/users`);

      // Формируем объект пользователей КБ, которые присутствуют в настройках телефонии (для которых настроена телефония)
      let users_info = [];
      for (let user_sip of settings_data.data.attributes.value.users) {
        if (user_sip.cb_user == CB.globals.user.id) continue;

        for (let user_cb of cb_users_data.data) {
          if (user_sip.cb_user == user_cb.id) {
            users_info.push({
              id: user_cb.id,
              name: user_cb.attributes.fio,
              short_number: user_sip.short_sip_number
            });
          }
        }
      }

      this.setStoreValue('users_info', users_info);

      this.login();

      window.addEventListener('storage', (function(event) {
        // совершить звонок на вкладке, где есть логин
        if (event.key == this.prefix + '_' + 'goCall') {
          if (JSON.parse(event.newValue) == true) {
            this.call();
            this.setStoreValue('goCall', false);
          }
        }

        // ответить на входящий звонок, где есть логин
        if (event.key == this.prefix + '_' + 'needAnswerCall') {
          if (JSON.parse(event.newValue) == true) {
            window.iSession.answer(this.options);
            this.setStoreValue('is_call', true);
            this.setStoreValue('needAnswerCall', false);
          }
        }

        // совершить mute звонка, где есть логин
        if (event.key == this.prefix + '_' + 'muteCall') {
          let flag = JSON.parse(event.newValue);
          this.muteCall(flag);
        }

        // совершить удержание звонка, где есть логин
        if (event.key == this.prefix + '_' + 'holdCall') {
          let flag = JSON.parse(event.newValue);
          this.holdCall(flag);
        }

        // начать переадресацию звонка, где есть логин (поставить на удержание)
        if (event.key == this.prefix + '_' + 'startReferCall') {
          let flag = JSON.parse(event.newValue);
          if(flag){
            this.holdCall();
          }
          this.setStoreValue('forwardCall', false);
        }

        // совершить переадресацию звонка, где есть логин (донабор и передресация звонка)
        if (event.key == this.prefix + '_' + 'forwardCall') {
          this.forwardCall();
          this.setStoreValue('startReferCall', false);
        }

        // если изменился на другой вкладке объект звонка, изменяем и на текущей
        if (event.key == this.prefix + '_' + 'Call') {
          let callEvent = JSON.parse(event.newValue);
          if (callEvent.numberForCall) { // номер телефона
            this.setStoreValue('phone_number', callEvent.numberForCall, false);
          }

          // информация об исходящем звонке
          if (callEvent.outgoing && !callEvent.incoming?.dropIncomingCall) {
            // если есть флаг, что нужно сбросить
            if (callEvent.outgoing.dropCall) {
              // если есть сессия, сбрасываем
              if (window.session) {
                JsSIP.Utils.closeMediaStream(window.localClonedStream);
                window.session.terminate();

                let new_call = callEvent;
                new_call.outgoing = {};
                this.setStoreValue('Call', new_call, true);
              }
              // очищаем инфу
              this.clearInfo(true, true);
            }
            // иначе просто записываем в хранилище объект звонка, без записи в localStorage
            else {
              this.setStoreValue('Call', callEvent, false);
            }

            // если номер исходящего звонка есть, то в свойство номер записываем его
            if (callEvent['outgoing'].number) {
              this.setStoreValue('phone_number', callEvent['outgoing'].number, false);
            }
          }
          // информация об входящем звонке
          else if (callEvent.incoming) {
            // если звонок принят
            if (callEvent['incoming'].incomingNumber && !callEvent['incoming'].incomingNumberGo) {
              this.setStoreValue('Call', callEvent, false);
            }

            if (callEvent.incoming.dropIncomingCall) { // если есть флаг, что нужно сбросить
              if (window.iSession && Object.keys(window.iSession).length > 0){ // есть сессия на текущей вкладке
                // Reject call (or hang up it)
                this.playSound('rejected.mp3', false);
                // callIndicator(true);
                let Call = {
                  isCall: false,
                  incoming: {},
                };
                this.setStoreValue('Call', Call);
                // Reject call (or hang up it)
                window.iSession.terminate();
                this.clearInfo(true, false);
                window.ua.unregister();
                window.iSession = null;
                window.session = null;
                window.location.reload();
              }
            } else {
              this.setStoreValue('Call', callEvent, false); // просто записываем данные о звонке
            }
          }

          // очищаем инфу
          if (
            !callEvent.isCall &&
            (callEvent.incoming && Object.keys(callEvent.incoming).length == 0 ||
            callEvent.outgoing && Object.keys(callEvent.outgoing).length == 0)
          ) {
            this.clearInfo(false, false);
          }
        }

        // все указанные ключи просто записываем в хранилище, без обновления localStorage
        if (
          event.key == this.prefix + '_' + 'asterisk_time' ||
          event.key == this.prefix + '_' + 'show_call_notification' ||
          event.key == this.prefix + '_' + 'notification_info' ||
          event.key == this.prefix + '_' + 'notification_text' ||
          event.key == this.prefix + '_' + 'notification_title' ||
          event.key == this.prefix + '_' + 'phone_number' ||
          event.key == this.prefix + '_' + 'service_text' ||
          event.key == this.prefix + '_' + 'is_call' ||
          event.key == this.prefix + '_' + 'call_type' ||
          event.key == this.prefix + '_' + 'call_status'
        ) {
          this.setStoreValue(event.key.replace((this.prefix + '_'), ''), JSON.parse(event.newValue), false);
        }
      }).bind(this));
    },

    methods: {
      /** получить данные */
      async getData(url) {
        let response = await fetch(url, {
          credentials: 'include',
          headers: {
            'X-Auth-Token': CB.globals.x_auth_token,
          },
        });
        let data = await response.json();
        return data;
      },

      /** авторизоваться */
      login() {
        if(
          (this.getStoreValue('provider_settings')?.sip_login || '').trim() == '' ||
          (this.getStoreValue('provider_settings')?.sip_password || '').trim() == ''
        ) {
          return;
        }

        window.socket = new JsSIP.WebSocketInterface(this.getStoreValue('provider_settings')?.server);

        window.ua = new JsSIP.UA({
          uri: "sip:" + this.getStoreValue('provider_settings').sip_login + "@" + this.getStoreValue('provider_settings').domain,
          password: this.getStoreValue('provider_settings').sip_password,
          display_name: this.prefix,
          sockets: [window.socket],
          session_timers: false
        });

        if (window.ua) {
          this.connected = true;
        }

        // соединяемся с сервером телефонии
        window.ua.on('connecting', () => console.log("UA connecting"));

        // соединились с сервером телефонии
        window.ua.on('connected', () => console.log("UA connected"));

        // сервер телефонии нас зарегистрировал, теперь можно звонить и принимать звонки
        window.ua.on('registered', () => {
          console.log("UA registered");
          window.ua['registeredStatus'] = true; // регистрация произошла
        });

        // сервер телефонии про нас больше не знает
        window.ua.on('unregistered', () => console.log("UA unregistered"));

        // сервер телефонии не зарегистрировал нас, что то не то, скорее всего неверный логин или пароль
        window.ua.on('registrationFailed', (data) => console.error("UA registrationFailed", data.cause));

        // заводим шарманку
        window.ua.start();

        setTimeout(() => {
          if ( // телефония примател, если регистрация sip пользователя не произошла
            typeof window.ua['registeredStatus'] === 'undefined' &&
            this.prefix === 'telephony_primatel_v2'
          ) location.reload(); // перезагружаю страницу, чтобы попробовать перезарегестрироваться
        }, 180001);

        // Вызываем функцию, которая на заведённую шарманку и её события накинет обработчики
        this.incomingCall();
      },

      /** выйти */
      logout() {
        console.log("on logout");
        // закрываем всё, вылогиниваемся из астера, закрываем коннект
        window.ua.unregister();
        this.connected = false;
        // Удаляем из хранилища переменную входящей сессиии
        window.iSession = null;
        // Удаляем из хранилища переменную исходящей сессиии
        window.session = null;
      },

      /** входящий звонок */
      incomingCall() {
        // событие создания сессии входящих звонков
        window.ua.on('newRTCSession', (async function(data) {
          let Call = this.getStoreValue('Call');

          // если уже есть инфа о звонке текущем, завершаем работу
          if (Call && Call.isCall) {
            return;
          }

          this.clearInfo(true);
          // записываем в хранилище сессию
          window.iSession = data.session;

          // если направление звонка - входящий
          if (window.iSession.direction === "incoming") {
            // вызываем функцию обработчика с пустым типом (для инициализации)
            this.incomingCallHandler('', data.request.from._display_name);
            try {
              let phone = data.request.from._display_name || data.request.from._uri._user;
              // заполняем карточку клиента
              await this.fillCardInfo(phone);
              this.setStoreValue('phone_number', phone);
            } catch (error) {
              console.error(error);
            }

            // крутим вертим всё по стримам звука, получаем стрим и кидаем его в элемент аудио на странице
            window.iSession.on('peerconnection', function(data) {
              data.peerconnection.addEventListener('addstream', function (e) {
                // set remote audio stream
                let remoteAudioControl = document.getElementById("remoteAudio");
                remoteAudioControl.srcObject = e.stream;
                remoteAudioControl.play();
              });
            });

            // Звонок принят, можно начинать говорить
            window.iSession.on('accepted', (function() {
              // console.log("UA session accepted");
              // Обработчик принятого входящего звонка
              this.incomingCallHandler('accepted');
            }).bind(this));

            window.iSession.on("confirmed",(function(data) {
              // console.log(data);
              // Событие непонятно когда наступает, да и обработчик пустой. По идее можно выпилить
              this.incomingCallHandler('confirmed');
            }).bind(this));

            window.iSession.on("ended",(function(data) {
              // console.log("session ended");
              // Обработчик окончания входящего звонка
              this.incomingCallHandler('ended');
            }).bind(this));

            window.iSession.on("failed",(function(data) {
              // console.log("session failed");
              // Обработчик ошибки входящего звонка
              this.incomingCallHandler('failed');
            }).bind(this));
          }
        }).bind(this));
      },

      /** обработчик входящих вызовов */
      incomingCallHandler(type = '', _phone_number = '') {
        // Звонок принят
        if (type == 'accepted') {
          this.stopSound();
          this.playSound('answered.mp3', false);
          let Call = this.getStoreValue('Call');
          Call['incoming']['incomingNumberGo'] = true;
          this.setStoreValue('Call', Call);
          this.setStoreValue('is_call', true);
          this.goTime();
        }
        else if (type == 'confirmed') {
        }
        // произошла ошибка
        else if (type == 'failed') {
          this.stopSound();
          let Call = this.getStoreValue('Call');
          Call.isCall = false;
          Call['incoming'] = {};
          this.setStoreValue('Call', Call);
          this.clearInfo(true);
          this.closeNotification();
        }
        // Звонок закончен
        else if (type == 'ended') {
          // store.get('stopSound')();
          let Call = this.getStoreValue('Call');
          Call.isCall = false;
          Call['incoming'] = {};
          this.setStoreValue('Call', Call);
          let time = this.getStoreValue('asterisk_time');
          this.requestForCreateTip(this.getStoreValue('phone_number'));
          this.clearInfo(true);
          this.showEndCallTime(time);
        }
        // При возникновении/инициации входящего звонка (когда кто-то тебе позвонил)
        else {
          this.stopSound();
          this.playSound('ringing.ogg', true);
          let Call = this.getStoreValue('Call') || {};
          Call['isCall'] = true;
          Call['incoming'] = {};
          Call['incoming']['incomingNumber'] = _phone_number;
          this.setStoreValue('Call', Call);
          this.setStoreValue('call_type', 'incoming');
          this.setStoreValue('notification_title', this.lang?.incoming_call || "");
          this.setStoreValue('phone_number', _phone_number);
          // this.setStoreValue('is_showed_call_block', true);
          this.setStoreValue('show_call_notification', true);
          window.opener.CB.store.dispatch(`${this.prefix}/setValue`, {        // INFO: события не доходят до родительского окна
            name: 'is_showed_call_block',                                   // поэтому вызываю родительскую ф-цию работы со store
            data: true,
            useLS: true,
          });
        }
      },

      /** обработчик исходящих вызовов */
      outgoingCallHandler(type = '', data = {}) {
        // Процесс дозвона
        if (type == 'progress') {
          this.setStoreValue('call_status', 'progress');
          this.setStoreValue('service_text', this.lang["progress___"]);
          this.playSound('ringback.ogg', true);
        }
        // Звонок принят
        else if (type == 'accepted') {
          this.setStoreValue('call_status', 'accepted');
          this.setStoreValue('service_text', 'Accepted');

          this.setStoreValue('show_number_block', true);

          this.stopSound();
          this.playSound('answered.mp3', false);

          this.goTime();
        }
        // Произошла ошибка
        else if (type == 'failed') {
          JsSIP.Utils.closeMediaStream(window.localClonedStream);

          this.setStoreValue('call_status', 'failed');
          this.setStoreValue('service_text', data.cause);

          this.stopSound();
          this.playSound('rejected.mp3', false);

          let Call = this.getStoreValue('Call');
          Call.isCall = false;
          Call['outgoing'] = {};
          this.setStoreValue('Call', Call);

          this.clearInfo(true);
          this.closeNotification();

          if (data.phone_numbers.length != 0) {
            this.call(data.phone_numbers);
            return;
          }
        }
        // Звонок окончен
        else if (type == 'ended') {
          JsSIP.Utils.closeMediaStream(window.localClonedStream);

          this.setStoreValue('call_status', 'ended');
          this.setStoreValue('service_text', this.lang["ended"]);

          this.stopSound();
          this.playSound('rejected.mp3', false);

          let Call = this.getStoreValue('Call');
          Call.isCall = false;
          Call['outgoing'] = {};
          this.setStoreValue('Call', Call);

          let time = this.getStoreValue('asterisk_time');
          this.requestForCreateTip(this.getStoreValue('phone_number'));
          this.clearInfo(true);
          this.showEndCallTime(time);
          //window.session.terminate();
          if (data.phone_numbers.length != 0) {
            this.call(data.phone_numbers);
            return;
          }
        }
        // Инициализация исходящего звонка
        else {
          this.setStoreValue('is_call', true);
          // Делаем ИСХОДЯЩИЙ звонок

          let Call = this.getStoreValue('Call') || {};
          Call['isCall'] = true;
          Call['outgoing'] = {};
          Call['outgoing']['number'] = data.number;
          this.setStoreValue('Call', Call);

          // callIndicator();

          this.setStoreValue('notification_title', this.lang["outgoing_call"]);
          this.setStoreValue('call_type', 'outgoing');
          this.setStoreValue('show_call_notification', true);
        }
      },

      /** Очистить информацию */
      clearInfo(is_hang_up = false, use_local_storage = true) {
        if (!is_hang_up) {
          this.setStoreValue('is_showed_call_block', false, use_local_storage);
          this.setStoreValue('phone_number', '', true);
        } else if (!document.hidden) {
          window.opener.CB.store.dispatch(`${this.prefix}/setValue`, {      // INFO: события не доходят до родительского окна
            name  : 'is_showed_call_block',                                 // поэтому вызываю родительскую ф-цию работы со store
            data  : true,
            useLS : true,
          });
        }
        this.setStoreValue('service_text', '', use_local_storage);
        this.setStoreValue('notification_text', '', use_local_storage);
        this.setStoreValue('is_call', false, use_local_storage);
        this.setStoreValue('call_type', '', use_local_storage);
        this.setStoreValue('asterisk_time', '00:00:00', use_local_storage);
        this.timer_id = clearInterval(this.timer_id);
        this.goTime(true);
        this.setStoreValue('redirect_number', '', use_local_storage);
        this.setStoreValue('show_redirect_block', false, use_local_storage);
      },

      /** время звонка */
      goTime(clear = false) {
        var prev_sec = 0,
          sec_,
          min_,
          hours_,
          zeros,
          zerom;

        if (clear) {
          if (this.getStoreValue('callInterval')) {
            setTimeout((function() {
              this.setStoreValue('callInterval', clearInterval(this.getStoreValue('callInterval')));
            }).bind(this), 0);
          } else {
            return false;
          }
        } else {
          let interval = setInterval((function() {
            sec_ = prev_sec % 60;
            min_ = parseInt(prev_sec / 60);
            hours_ = '00';

            zeros = '';
            if (sec_ < 10) zeros = '0';
            zerom = '';
            if (min_ < 10) zerom = '0';

            this.setStoreValue('asterisk_time', hours_ + ':' + zerom + min_ + ':' + zeros + sec_);

            prev_sec++;
          }).bind(this), 1000);

          this.setStoreValue('callInterval', interval);
        }
      },

      /** закрыть уведомление */
      closeNotification() {
        this.setStoreValue('show_call_notification', false);
        this.setStoreValue('notification_title', '');
        this.setStoreValue('notification_text', '');
        //this.setStoreValue('notification_info', {});
      },

      /** запрос создания напоминания */
      async requestForCreateTip(_phone_number) {
        return;
        await fetch(`../../api/dev/cbext/clientbase/iptel_${this.name}/createAnsweredTip?user_id=${CB.globals.user.id}&number=${_phone_number}`, {
          method: 'GET',
          credentials: 'include',
          redirect: 'follow',
          headers: {
            'X-Auth-Token': CB.globals.x_auth_token,
            "Accept": "application/vnd.api+json",
            "Content-Type": "application/vnd.api+json"
          }
        });
      },

      /** показать время окончания разговора */
      showEndCallTime(time) {
        this.setStoreValue('show_call_notification', true);
        this.setStoreValue('notification_title', this.lang["the_call_is_over"]);
        this.setStoreValue('notification_text', `${this.lang["call_duration"]} ${time}`);
      },

      /** заполнить карточку клиента */
      async fillCardInfo(_phone_number) {
        let response = await fetch(`../../api/dev/cbext/clientbase/iptel_${this.name}/find?number=${_phone_number}`, {
          method: 'GET',
          credentials: 'include',
          redirect: 'follow',
          headers: {
            'X-Auth-Token': CB.globals.x_auth_token || window.x_auth_token,
            "Accept": "application/vnd.api+json",
            "Content-Type": "application/vnd.api+json"
          }
        });
        let msg = await response.json();
        if (msg) {
          window.opener.CB.store.dispatch(`${this.prefix}/setValue`, {
            name  : 'show_call_notification',
            data  : false,
            useLS : true,
          });
          window.opener.CB.store.dispatch(`${this.prefix}/setValue`, {
            "name": "notification_info",
            "data": {},
            "useLS": true
          });

          // this.setStoreValue('show_call_notification', true);
          window.opener.CB.store.dispatch(`${this.prefix}/setValue`, {      // INFO: события не доходят до родительского окна
            name  : 'show_call_notification',                               // поэтому вызываю родительскую ф-цию работы со store
            data  : true,
            useLS : true,
          });
          var from = msg['name']['field_name'] == "" ? "" : `${msg['name']['field_name']}: `;
          var from_name = msg['name']['value'];
          if (from_name == 'NULL_NAME') from_name = '';
          var from_url = msg['url'];

          let addit = [];
          // дополнительные поля
          if (msg['issetClientCard']) {
            let generateAdditionalField = field_data => {
              let text = "";
              if (field_data.field_name != "") {
                text = `${field_data.field_name}:`
              }
              return text != "" ? `${text} ${field_data.value}` : "";
            }
            msg.additional_fields.forEach(item => addit.push(generateAdditionalField(item)));
          }

          let clientCardInfo = {               // INFO: события не доходят до родительского окна
            name: 'notification_info',         // поэтому вызываю родительскую ф-цию работы со store
            data: {
              from_number: _phone_number,
              from,
              from_name,
              from_url,
              addit_1: addit[0],
              addit_2: addit[1],
              addit_3: addit[2]
            },
            useLS: true,
          };
          console.log(clientCardInfo);
          window.opener.CB.store.dispatch(`${this.prefix}/setValue`, clientCardInfo);
        }
      },

      /** хз, запрос звонка */
      async callRequest(params, command, data = null, mode = 'json') {
        // Сформировать URL
        let url = params['url']
          + '?svc=' + command
          + '&mode=' + mode // не обязательный, по умолчанию будет 'xml'
          + '&lang=ru';      // не обязательный, по умолчанию будет 'en'

        let req = {
          method : 'POST',
          body: JSON.stringify(data, null, 4),
          credentials: 'same-origin', // enable cookies
        };

        let response = await fetch(url, req);

        let contentTypeRaw = response.headers.get("content-type");

        let A = contentTypeRaw.split(/;/);

        let contentType = A ? A[0] : contentTypeRaw;

        switch (contentType) {
          case 'application/json':
            let json = await response.json();
            if (response.ok) {
              return json;
            } else {
              throw new Error('[' + command + '] Call failed. HTTP status ' + response.status + ' ' + response.statusText);
            }
          case 'audio/mpeg':
            let blob = await response.blob();
            if (response.ok) {
              return window.URL.createObjectURL(blob);
            } else {
              throw new Error('[' + command + '] Call failed. HTTP status ' + response.status + ' ' + response.statusText);
            }
          default:
            return undefined;
        }
      },

      /** звонок */
      async call(_phone_numbers = []) {
        this.clearInfo(true);
        this.closeNotification();
        //this.setStoreValue('show_call_notification', true);

        let phone_numbers = _phone_numbers;
        if (phone_numbers.length != 0) {
          // получаем первый номер из массива переданных номеров (может быть несколько, для автообзвона)
          this.setStoreValue('phone_number', phone_numbers.shift().trim());
        }

        let number = this.getStoreValue('phone_number'); // Берём номер
        if (number === void(undefined)) {
          return;
        }

        // Проверяем номер на валидность
        if (number.trim().length == 0) {
          this.setStoreValue('service_text', 'Number is empty!');
          setTimeout(() => {
            this.setStoreValue('service_text', '');
          }, 5000);
          return;
        }

        if(number.length >= 10 && number[0] == '+' && number[1] == '7'){
          number = '8' + number.slice(2);
        }

        try {
          // Заполняем карточку клиента
          await this.fillCardInfo(number);
        } catch (error) {
          console.error(error);
        }

        // если есть объект jssip для исходящих звонков (он при логине создаётся)
        if (window.ua) {
          // Вызываем метод call и если записываем полученную сессию исходящего звонка
          window.session = window.ua.call(number, this.options);

          // Обработчик при инициализации исходящего звонка
          this.outgoingCallHandler('', { number });

          // Астер нас соединил с абонентом
          window.session.on('connecting', (function() {
            this.setStoreValue('service_text', 'Connecting...');
            this.playSound("ringback.ogg", true);

            // Тут мы подключаемся к микрофону и цепляем к нему поток, который пойдёт в астер
            let peerconnection = window.session.connection;
            window.localStream = peerconnection.getLocalStreams()[0];
            if (window.localStream) {
              let localStream = window.localStream;
              window.localClonedStream = localStream.clone();
              let localAudioControl = document.getElementById("localAudio");
              localAudioControl.srcObject = window.localClonedStream;
            }

            // Как только астер отдаст нам поток абонента, мы его засунем к себе в наушники
            peerconnection.addEventListener('addstream', (event) => {
              let remoteAudioControl = document.getElementById("remoteAudio");
              remoteAudioControl.srcObject = event.stream;
              remoteAudioControl.play();
            });
          }).bind(this));

          // В процессе дозвона
          window.session.on('progress', (function(){
            // console.log("UA session progress");
            this.outgoingCallHandler('progress');
          }).bind(this));

          // Дозвон завершился неудачно, например, абонент сбросил звонок
          window.session.on('failed', (function(data){
            // console.log("UA session failed");
            // console.log(data);
            this.outgoingCallHandler('failed', {
              cause: data.cause,
              phone_numbers
            });
          }).bind(this));

          // Поговорили, разбежались
          window.session.on('ended', (function(data){
            // console.log("UA session ended");

            this.outgoingCallHandler('ended', {
              phone_numbers
            });
          }).bind(this));

          // Звонок принят, моно начинать говорить
          window.session.on('accepted', (function(){
            // console.log("UA session accepted");
            this.outgoingCallHandler('accepted');
          }).bind(this));
        }
      },

      /** входящий повесить трубку */
      incomingHangUp() {
        this.playSound("rejected.mp3", false); // Проигрываем звук окончания
        window.iSession.terminate(); // Заканчиваем звонок
      },

      /** Вешать трубку */
      hangUp() {
        this.setStoreValue('is_call', false);
        let Call = {
          isCall: false,
          outgoing: {},
        };
        this.setStoreValue('Call', Call);
        window.session.terminate();
      },

      /** отключить звук вызова */
      muteCall(need_mute) {
        let sess = this.getStoreValue('call_type') != 'outgoing' ? window.session : window.iSession;
        if (need_mute) {
          sess.mute();
        } else {
          sess.unmute();
        }
      },

      /** удерживать вызов */
      holdCall() {
        window.iSession.hold();
      },

      /** переадресация звонка */
      forwardCall() {
        let redirect_number = String(localStorage.getItem(`${options.prefix}_redirect_number`)).replace(/[^0-9]/g, '');
        window.iSession.refer(redirect_number);
      },

      /** играть звук */
      playSound(soundName, loop) {
        if (
          options.prefix != "telephony_uiscom" &&
          soundName != "ringback.ogg"
        ) {
          let sound_element = document.getElementById('sounds');
          sound_element.src = `./sounds/${soundName}`;
          sound_element.loop = loop;
          sound_element.muted = true;
          sound_element.play();
          sound_element.muted = false;
        }
      },

      /** остановить звук */
      stopSound() {
        let sound_element = document.getElementById('sounds');
        sound_element.pause();
        sound_element.currentTime = 0.0;
      },
    },

    template: `
      <div>
        <div>
          <div style="font-weight: bold">{{lang['Connection']}}:</div>
          <div v-if="connected != null && connected">{{lang['Telephony_module_connected']}}</div>
          <div v-else>{{lang['Connection_failure']}}</div>
        </div>

        <div>
          <div style="font-weight: bold">{{lang['Current_call']}}:</div>
          <div v-if="connected != null && connected && $store.state[prefix].Call && $store.state[prefix].Call.isCall">
            {{lang['Made_call']}} {{$store.state[prefix].service_text}} {{$store.state[prefix].asterisk_time}}
          </div>
          <div v-else>{{lang['No_made_call']}}</div>
        </div>

        <div>
          <div style="font-weight: bold">{{lang['Last_call_made']}}:</div>
          <div>{{$store.state[prefix].notification_text}}</div>
        </div>
      </div>
    `
  }
}
