import Label from "../../../shared/Label/Label.js";
import FormField from "../../../entities/FormLine/FormField/FormField.js";
import useScopedStyleMixin from '../../../useScopedStyleMixin.js';
import CalcConditionsStyles from './CalcConditionsStyles.js';

const BASE_CONDITION_OPTIONS = [
  { id: 0, name: '---' },
  { id: 2, name: lang.Show_field },
  { id: 3, name: lang.Field_changed },
  { id: 4, name: lang.Delete_in_table },
  { id: 5, name: lang.Import_in_table },
  { id: 7, name: lang.Restore_in_table },
  { id: 8, name: lang.On_schedule }
];
const PERIOD_OPTIONS = [
  { id: '* * * * *', name: 'Ежеминутно' },
  { id: '1,15,30,45 * * * *', name: 'Каждые 15 минут' },
  { id: '1-59/5 * * * *', name: 'Каждые 5 минут' },
  { id: '1 12 * * *', name: 'Каждый день в 12 часов' },
  { id: '1 23 * * *', name: 'Каждый день в 23 часа' },
  { id: '1 3 * * *', name: 'Каждый день в 3 часа' },
  { id: '1 * * * *', name: 'Каждый час' },
  { id: 'advanced', name: 'Расширенный' }
];
const DEFAULT_PERIOD = PERIOD_OPTIONS.find(({name}) => name === 'Каждый час').id

const CalcConditions = {
  name: "calc-conditions",

  components: {
    'Dropdown': primevue.dropdown,
    Label,
    FormField
  },

  props: {
    settings: Object,
    dataSource: Object,
    fieldsSource: Object,
    data: Object
  },

  mixins: [useScopedStyleMixin, CalcConditionsStyles],
  setup(props, { emit }) {
    let fieldCond = props.settings.fieldCond;
    let fieldPeriod = props.settings.fieldPeriod;
    let tableFields = {
      "param": {
        "type": "dropdown",
        "optionsFrom": {
          "source": "fields",
          "parentId": 0
        },
        "required": true
      }
    };

    return {
      fieldCond,
      fieldPeriod,
      tableFields,
    }
  },

  data() {
    return {
      isValidPeriod: true,
      conditions: [],
      componentClass: 'calc-conditions',
      periodOptions: PERIOD_OPTIONS,
      cronStartActivated: false,
      showAdvanced: false,
    }
  },

  methods: {
    validateCron() {
      const cronRegex = /^(\*|([0-5]?\d(-[0-5]?\d)?|([0-5]?\d(,[0-5]?\d)*))|\*\/[0-5]?\d)\s+(\*|([0-2]?\d(-[0-2]?\d)?|([0-2]?\d(,[0-2]?\d)*))|\*\/[0-2]?\d)\s+(\*|([1-2]?\d|3[0-1])(-[1-2]?\d|3[0-1])?|([1-2]?\d|3[0-1])(,[1-2]?\d|3[0-1])*|\*\/[1-2]?\d|3[0-1])\s+(\*|([1]?\d(-[1]?\d)?|([1]?\d(,[1]?\d)*))|\*\/[1]?\d)\s+(\*|([0-7](-[0-7])?|([0-7](,[0-7])*))|\*\/[0-7])$/;
      this.isValidPeriod = cronRegex.test(this.dataSource['cron_id']['period']);
    },
    updateConditionTypePeriod(value) {
      this.periodSelected = null;
      if (value !== 0) this.addConditionType(value);
    },
    updateConditionType(value, index) {
      if (!this.dataSource[this.fieldCond]) {
        this.dataSource[this.fieldCond] = [];
      }
  
      if (value === 8) {
        // Удаляем элемент, если выбран пустой
        this.dataSource[this.fieldCond].splice(index, 1);
        this.addConditionTypePeriod();
      } else if (value === 0) {
        // Удаляем элемент, если выбран пустой
        this.dataSource[this.fieldCond].splice(index, 1);
      } else {
        // Обновляем тип без мутации других свойств
        this.dataSource[this.fieldCond][index].type = value;
      }
    },
    addConditionTypePeriod() {
      this.dataSource[this.fieldPeriod] = this.periodSelected || DEFAULT_PERIOD;
    },
    addConditionType(value) {
      if(value === 8)
        return this.addConditionTypePeriod();
      if (!this.dataSource[this.fieldCond]) {
        this.dataSource[this.fieldCond] = [];
      }
  
      // Добавляем новый элемент
      this.dataSource[this.fieldCond].push({ type: value });
    },
  },
  computed: {
    hasType() {
      return (type) => this.filteredConditions.some(item => item.type === type);
    },
    conditionTypeOptions() {
      return (condition) => BASE_CONDITION_OPTIONS.map(option => ({
        ...option,
        disabled: (
          (
            [4, 5, 7].includes(option.id) && 
            option.id !== condition.type && 
            this.hasType(option.id)
          ) ||
          (
            option.id === 8 && this.hasPeriodConditionType
          )) ||
          (condition.type === 0 && option.id === 0)
      }));
    },
    filteredConditions() {
      return this.dataSource[this.fieldCond] || [];
    },
    parentId() {
      if (!this.data[this.settings.from_list]) return null;
  
      const tableMap = this.data[this.settings.from_list].find(
        item => item.id == this.dataSource[this.settings.from_list_value]
      );
  
      return tableMap ? tableMap.internal_object_id : null;
    },
    hasPeriodConditionType() {
      return !!this.dataSource[this.fieldPeriod];
    },
    periodSelected: {
      get() {
        if(this.showAdvanced)
          return 'advanced';

        const period = this.dataSource[this.fieldPeriod];
        const hasId = this.periodOptions.some(option => option.id === period);

        if(!period)
          return this.dataSource[this.fieldPeriod] = DEFAULT_PERIOD;

        if(!hasId) {
          this.showAdvanced = true;
          return 'advanced';
        }
        return period;
      },
      set(value) {
        this.isValidPeriod = true;
        if(value === 'advanced')
          return this.showAdvanced = true;
        this.showAdvanced = false;
        this.dataSource[this.fieldPeriod] = value;
      }
    },
  },
  watch: {
    parentId: {
      handler(newId) {
        if (newId !== null) {
          this.tableFields.param.optionsFrom.parentId = newId;
        }
      },
      immediate: true
    }
  },
  template: /*html*/`
    <section :class="componentClass + hash">
      <div :class="componentClass + '__label' + hash">
        <Label :label="settings.label" :help="settings.help" />
      </div>
      <div 
        :class="componentClass + '__items' + hash"
      >
        <div 
          v-for="(condition, ident) in filteredConditions"
          :class="[componentClass + '__item' + hash]"
        >
          <Dropdown 
            v-model="condition.type"
            optionLabel="name"
            optionValue="id"
            optionDisabled="disabled"
            :options="conditionTypeOptions(condition)"
            :class="componentClass + '__condition-type-selector' + hash"
            @update:modelValue="updateConditionType($event, ident)"
          />
          <template v-if="(condition['type'] == 2) || (condition['type'] == 3)">
            <FormField
              :class="componentClass + '__condition-form-field-selector' + hash"
              :dataSource="condition" 
              :fieldsSource="tableFields"  
              :storeFieldId="'param'"
              :table="parentId"
            />
          </template>
        </div>
        <div
          v-if="hasPeriodConditionType"
          :class="[componentClass + '__item' + hash]"
        >
          <Dropdown 
            :modelValue="8"
            optionLabel="name"
            optionValue="id"
            optionDisabled="disabled"
            :options="conditionTypeOptions({ type: 8 })"
            :class="componentClass + '__condition-type-selector' + hash"
            @update:modelValue="updateConditionTypePeriod($event)"
          />
          <Dropdown 
            v-model="periodSelected"
            optionLabel="name"
            optionValue="id"
            :class="componentClass + '__condition-interval-selector' + hash"
            :options="periodOptions"
          />
          <input
            v-if="periodSelected == 'advanced'" 
            type="text" 
            @input="validateCron" 
            :style="{ width: '80px', backgroundColor: isValidPeriod ? 'white' : 'rgb(250, 167, 167)' }"
            v-model="dataSource[fieldPeriod]" 
            class="form-control"
          />
          
        </div>
        <div
          :class="[componentClass + '__item' + hash]"
        >
          <Dropdown 
            :modelValue="0"
            optionLabel="name"
            optionValue="id"
            optionDisabled="disabled"
            :options="conditionTypeOptions({ type: 0 })"
            :class="componentClass + '__condition-type-selector' + hash"
            @update:modelValue="addConditionType($event)"
          />
        </div>
      </div>
    </section>
  `,
};

export default CalcConditions;