//
// Хранилище записей пользовательских таблиц
//

/**
 * Хранилище (pinia-store) записей пользовательских таблиц (ресурсов)
 */
export const useRecordsStore = Pinia.defineStore('recordsStore', () => {
  const selfee = this
  const graph = Vue.ref({})

  /**
   * Найти запись (ресурс) в хранилище по указателю ресурса
   *
   * Универсальный поиск:
   * * find(type, id) - найти запись типа type с идентификатором id
   * * find({type, id}) - найти запись по указателю ресурса
   * * find([{type, id}, ...]) - найти массив записей по указателю ресурса
   * @param {string} type либо тип ресурса (идентификатор таблицы) либо указатель ресурса, либо массив указателей ресурса
   * @param {any} id идентификатор записи
   * @return {{}|[]} найденная запись или массив найденных записей
   */
  function find(type, id = null) {
    if (type === null) {
      return null;
    }

    if (Array.isArray(type)) {
      return type.map((identifier) => find(identifier));
    }

    if (typeof type === 'object') {
      return find(type.type, type.id);
    }

    return (graph.value[type] && graph.value[type][id]) || null;
  }

  /**
   * Найти все записи (ресурсы) указанного типа в хранилище
   */
  function findAll(type) {
    if (!graph.value[type]) {
      return [];
    }

    return Object.keys(graph.value[type]).map((id) => graph.value[type][id]);
  }

  /** DEPRECATED get all records from store */
  const getgraph = Vue.computed( () => () => {
    return graph.value;
  })

  /**
   * Синхронизировать транспортный JSONAPI-ответ сервера (может быть документ или массив документов) с хранилищем
   *
   * @param {Object} document JSONAPI-ответ, поступивший с jsonapi-сервера
   * @returns
   */
  function sync(document) {
    if ('included' in document) {
      document.included.map(syncResource);
    }

    return Array.isArray(document.data) ? document.data.map(syncResource) : syncResource(document.data);
  }

  /**
   * Выполнить запрос к JSONAPI-серверу
   *
   * @param {string} entrypoint запрашиваемый API-ресурс
   * @param {Object} options настройки запроса
   * @returns {Object} {response, document, data }
   */
  async function query(entrypoint, options = {}) {
    options.headers = options.headers || {};
    options.headers['Accept'] = 'application/vnd.api+json';
    options.headers['x-auth-token'] = window.x_auth_token;

    if (options.body) {
        options.body = JSON.stringify(options.body);
        options.headers['Content-Type'] = 'application/vnd.api+json';
    }

    const response = await fetch('./api/dev/' + entrypoint, options);
    if (response.status === 204) {
      return { response };
    } else {
      const document = await response.json();
      const data = sync(document);
      return { response, document, data };
    }
  }

  /**
   * Синхронизировать JSONAPI-документ с хранилищем
   */
  function syncResource(data) {
    const { type, id } = data;
    graph.value[type] = graph.value[type] || {};

    if (graph.value[type][id]) {
        graph.value[type][id].merge(data);
    } else {
        graph.value[type][id] = new Record(data, selfee);
    }

    return graph.value[type][id];
  }

  /**
   * Сохранить на сервер запись (ресурс) из хранилища
   *
   * Тестовая реализация (вчерную).
   * Выполняется PATCH, то есть сейчас реализовано только для существующих записей.
   *
   * @param {{type, id}} item ресурсный идентификатор записи, которую следует сохранить на сервер
   */
  async function persist(item) {
    const record = find(item)
    return query(`${item.type}/${item.id}`, {
      method: 'PATCH',
      body: {
        data: {
          type: record.type,
          id: record.id,
          attributes: record.getAttributes(),
        }
      }
    })
  }

  /**
   * Удалить запись (ресурс) из хранилища
   *
   * @param {{type, id}} item ресурсный идентификатор записи, которую следует удалить
   */
  function forget(item) {
    delete graph.value[item.type][item.id];
  }

  /**
   * Удалить все записи (ресурсы) из хранилища
   */
  function reset() {
    graph.value = {}
  }

  return {
    query,
    find,
    findAll,
    persist,
    forget,
    reset,
    getgraph,
  }
})

/**
 * Запись пользовательской таблицы
 */
 export class Record {
  type;
  id;
  _jsonapi;

  /**
   * Создать объект-пользовательскую запись
   *
   * @param {any} data массив данных JSONAPI-документа
   */
  constructor(data) {
    this._jsonapi = {
      attributes: [],
      relationships: {},
      meta: {},
      links: {},
    }
    this.merge(data)
  }

  getAttributes() {
    const result = {}
    this._jsonapi.attributes.forEach((name) => {
      result[name] = this[name]
    })
    return result
  }

  getRelationship(name) {
    const data = this._jsonapi.relationships[name].data
    const store = useRecordsStore()
    return store.find(data)
  }

  identifier() {
    return {
      id: this.id,
      type: this.type
    }
  }

  merge(data) {
    if ("type" in data) {
      this.type = data.type
    }

    if ("id" in data) {
      this.id = data.id
    }

    if ("attributes" in data) {
      this._jsonapi.attributes = Object.keys(data.attributes)
      this._jsonapi.attributes.forEach((name) => {
        if (['id','type','_jsonapi'].includes(name)) {
          return
        }
        this[name] = data.attributes[name]
      })
    }

    if ("relationships" in data) {
      Object.entries(data.relationships).forEach(([name, relationship]) => {
        this._jsonapi.relationships[name] = this._jsonapi.relationships[name] || {};
        Object.assign(this._jsonapi.relationships[name], relationship);
        if (Object.getOwnPropertyDescriptor(Object.getPrototypeOf(this), name) || Object.getOwnPropertyDescriptor(this, name)) {
          return;
        }
        this[name] = null;
        Object.defineProperty(this, name, {
          get: () => this.getRelationship(name),
          configurable: true,
          enumerable: true,
        });
      });
    }

    if ("links" in data) {
      this._jsonapi.links = data.links
    }

    if ("meta" in data) {
      this._jsonapi.meta = data.meta
    }
  }
}
